<?php

/**
 * JCH Optimize - Performs several front-end optimizations for fast downloads
 *
 *  @package   jchoptimize/core
 *  @author    Samuel Marshall <samuel@jch-optimize.net>
 *  @copyright Copyright (c) 2024 Samuel Marshall / JCH Optimize
 *  @license   GNU/GPLv3, or later. See LICENSE file
 *
 *  If LICENSE file missing, see <http://www.gnu.org/licenses/>.
 */

namespace CodeAlfa\Component\JchOptimize\Administrator\Service\Provider;

use Joomla\CMS\Cache\CacheControllerFactoryAwareInterface;
use Joomla\CMS\Cache\CacheControllerFactoryInterface;
use Joomla\CMS\Factory;
use Joomla\CMS\Form\FormFactoryAwareInterface;
use Joomla\CMS\Form\FormFactoryInterface;
use Joomla\CMS\Mail\MailerFactoryAwareInterface;
use Joomla\CMS\Mail\MailerFactoryInterface;
use Joomla\CMS\MVC\Factory\ApiMVCFactory;
use Joomla\CMS\MVC\Factory\MVCFactoryInterface;
use Joomla\CMS\Router\SiteRouter;
use Joomla\CMS\Router\SiteRouterAwareInterface;
use Joomla\CMS\User\UserFactoryAwareInterface;
use Joomla\CMS\User\UserFactoryInterface;
use Joomla\Database\DatabaseAwareInterface;
use Joomla\Database\DatabaseInterface;
use Joomla\DI\Container;
use Joomla\DI\ContainerAwareInterface;
use Joomla\DI\ServiceProviderInterface;
use Joomla\Event\DispatcherAwareInterface;
use Joomla\Event\DispatcherInterface;

use function defined;

// phpcs:disable PSR1.Files.SideEffects
defined('_JEXEC') or die;
// phpcs:e_JEXECes.SideEffects

/**
 * Service provider for the service MVC factory.
 *
 * @since  4.0.0
 */
class MVCContainerFactory implements ServiceProviderInterface
{
    /**
     * The extension namespace
     *
     * @var  string
     *
     * @since   4.0.0
     */
    private $namespace;

    /**
     * MVCFactory constructor.
     *
     * @param string $namespace The namespace
     *
     * @since   4.0.0
     */
    public function __construct(string $namespace)
    {
        $this->namespace = $namespace;
    }

    /**
     * Registers the service provider with a DI container.
     *
     * @param Container $container The DI container.
     *
     * @return  void
     *
     * @since   4.0.0
     */
    public function register(Container $container)
    {
        $container->set(
            MVCFactoryInterface::class,
            function (Container $container) {
                if (Factory::getApplication()->isClient('api')) {
                    $factory = new ApiMVCFactory($this->namespace);
                } else {
                    $factory = new \CodeAlfa\Component\JchOptimize\Administrator\Extension\MVCContainerFactory($this->namespace);
                }

                if ($factory instanceof ContainerAwareInterface) {
                    $factory->setContainer($container);
                }
                if ($factory instanceof FormFactoryAwareInterface) {
                    $factory->setFormFactory($container->get(FormFactoryInterface::class));
                }
                if ($factory instanceof DispatcherAwareInterface) {
                    $factory->setDispatcher($container->get(DispatcherInterface::class));
                }
                if ($factory instanceof DatabaseAwareInterface) {
                    $factory->setDatabase($container->get(DatabaseInterface::class));
                }
                if ($factory instanceof SiteRouterAwareInterface) {
                    $factory->setSiteRouter($container->get(SiteRouter::class));
                }
                if ($factory instanceof CacheControllerFactoryAwareInterface) {
                    $factory->setCacheControllerFactory($container->get(CacheControllerFactoryInterface::class));
                }
                if ($factory instanceof UserFactoryAwareInterface) {
                    $factory->setUserFactory($container->get(UserFactoryInterface::class));
                }
                if ($factory instanceof MailerFactoryAwareInterface) {
                    $factory->setMailerFactory($container->get(MailerFactoryInterface::class));
                }

                return $factory;
            }
        );
    }
}
