<?php

/**
 * JCH Optimize - Performs several front-end optimizations for fast downloads
 *
 * @package   jchoptimize/joomla-platform
 * @author    Samuel Marshall <samuel@jch-optimize.net>
 * @copyright Copyright (c) 2023 Samuel Marshall / JCH Optimize
 * @license   GNU/GPLv3, or later. See LICENSE file
 *
 *  If LICENSE file missing, see <http://www.gnu.org/licenses/>.
 */

namespace CodeAlfa\Plugin\System\JchPageCache\Extension;

use CodeAlfa\Component\JchOptimize\Administrator\Extension\JchOptimizeComponent;
use CodeAlfa\Minify\Js;
use Exception;
use JchOptimize\Core\Helper;
use JchOptimize\Core\Html\HtmlElementBuilder;
use JchOptimize\Core\PageCache\PageCache;
use JchOptimize\Core\Platform\PathsInterface;
use JchOptimize\Core\SystemUri;
use Joomla\CMS\Application\CMSApplication;
use Joomla\CMS\Application\CMSApplicationInterface;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\Session\Session;
use Joomla\DI\Container;
use Joomla\Event\Priority;
use Joomla\Event\SubscriberInterface;

use function defined;
use function str_replace;

// phpcs:disable PSR1.Files.SideEffects
defined('_JEXEC') or die('Restricted Access');
// phpcs:enable PSR1.Files.SideEffects

class JchPageCache extends CMSPlugin implements SubscriberInterface
{
    public bool $enabled = false;

    private CMSApplication|CMSApplicationInterface|null $app;

    private PageCache $pageCache;

    private Container $container;

    public static function getSubscribedEvents(): array
    {
        return [
            'onAfterInitialise' => 'onAfterInitialise',
            'onAfterRoute' => 'onAfterRoute',
            'onAfterRender' => ['onAfterRender', Priority::LOW],
            'onAfterRespond' => ['onAfterRespond', Priority::LOW],
            'onPageCacheSetCaching' => 'onPageCacheSetCaching',
            'onPageCacheGetKey' => 'onPageCacheGetKey',
            'onAjaxGetformtoken' => 'onAjaxGetformtoken',
            'onAjaxUpdatehits' => 'onAjaxUpdatehits',
        ];
    }

    public function onAfterInitialise(): void
    {
        $this->app = $this->getApplication();
        $input = $this->app->getInput();

        if ($this->app === null) {
            return;
        }

        if (!$this->app->isClient('site')) {
            return;
        }

        if ($this->app->get('offline', '0')) {
            return;
        }

        if ($this->app->getMessageQueue()) {
            return;
        }

        //Disable if the component is not installed or disabled
        if (!ComponentHelper::isEnabled('com_jchoptimize')) {
            return;
        }

        /** @var JchOptimizeComponent $component */
        $component = $this->app->bootComponent('com_jchoptimize');
        $this->container = $component->getContainer();

        //Disable if jchnooptimize set
        if (
            $input->get('jchnooptimize') == '1'
            || $input->get('jchbackend') == '1'
        ) {
            return;
        }

        //Disable if we couldn't get cache object
        try {
            $this->pageCache = $this->container->get(PageCache::class);
        } catch (Exception) {
            return;
        }

        if (JDEBUG) {
            $this->pageCache->disableCaptureCache();
        }

        $this->enabled = true;

        try {
            $this->pageCache->initialize();
        } catch (Exception) {
            $this->enabled = false;

            return;
        }
    }

    public function onAfterRoute(): void
    {
        if (!$this->enabled) {
            return;
        }

        $this->pageCache->outputPageCache();
        //If we're forcing ssl on the front end but not serving https, disable caching
        if ($this->app->get('force_ssl') === 2 && SystemUri::currentUri()->getScheme() !== 'https') {
            $this->enabled = false;
            return;
        }
        try {
            $excludedMenus = $this->container->get('params')->get('cache_exclude_menu', []);
            $excludedComponents = $this->container->get('params')->get('cache_exclude_component', []);

            if (
                in_array($this->app->getInput()->get('Itemid', '', 'int'), $excludedMenus)
                || in_array($this->app->getInput()->get('option', ''), $excludedComponents)
            ) {
                $this->enabled = false;
                $this->pageCache->disableCaching();

                return;
            }
            //Now may be a good time to set Caching
            $this->pageCache->setCaching();
        } catch (Exception $e) {
        }
    }

    public function onAfterRender(): void
    {
        if (!$this->enabled) {
            return;
        }

        if (!Helper::validateHtml($this->app->getBody())) {
            $this->pageCache->disableCaching();

            return;
        }

        //Disable gzip so the HTML can be cached later
        $this->app->set('gzip', false);
    }

    public function onAfterRespond(): void
    {
        //Page cache could be disabled at runtime so check again here
        if ($this->enabled && $this->app instanceof CMSApplication && $this->pageCache->getCachingEnabled()) {
            $body = $this->app->getBody();
            //Still need to validate the HTMl here. We may be on a redirect.
            if (Helper::validateHtml($body)) {
                $this->pageCache->store($this->addUpdateHitScript($this->addUpdateFormTokenAjax($body)));
            }
        }
    }

    /**
     * If Page Cache plugin is already disabled then this will disable the Page Cache object when it is constructed
     */
    public function onPageCacheSetCaching($event): void
    {
        $event->addArgument('result', [$this->enabled]);
    }

    public function onPageCacheGetKey($event): void
    {
        $event->addArgument('result', [$this->app->getLanguage()->getTag()]);
    }

    private function addUpdateFormTokenAjax(string $html): string
    {
        if (!$this->pageCache->isCaptureCacheEnabled()) {
            return $html;
        }

        $url = SystemUri::siteBaseFull(
            $this->container->get(PathsInterface::class)
        ) . 'index.php?option=com_ajax&format=json&plugin=getformtoken';

        /** @see JchPageCache::onAjaxGetformtoken() */
        $script = <<<JS
let jchCsrfToken;

const updateFormToken = async() => {
    const response = await fetch('$url');
    
    if (response.ok) {
        const jsonValue = await response.json();
            
        return Promise.resolve(jsonValue);
    }
}

updateFormToken().then(data => {
    const formRegex = new RegExp('[0-9a-f]{32}');
    jchCsrfToken = data.data[0];
    
    for (let formToken of document.querySelectorAll('input[type=hidden]')){
        if (formToken.value == '1' && formRegex.test(formToken.name)){
            formToken.name = jchCsrfToken;
        }
    }
    
    const jsonRegex = new RegExp('"csrf\.token":"[^"]+"');
    
    for(let scriptToken of document.querySelectorAll('script[type="application/json"]')){
        if(scriptToken.classList.contains('joomla-script-options')){
            let json = scriptToken.textContent;
            if(jsonRegex.test(json)){
                scriptToken.textContent = json.replace(jsonRegex, '"csrf.token":"' + jchCsrfToken + '"');
            }
        }
    }
    
    updateJoomlaOption();
});

function updateJoomlaOption(){
    if (typeof Joomla !== "undefined" ){
        Joomla.loadOptions({"csrf.token": null});
        Joomla.loadOptions({"csrf.token": jchCsrfToken});
    }
}

document.addEventListener('onJchJsDynamicLoaded', (event) => {
    updateJoomlaOption();
});
JS;
        $htmlScript = HtmlElementBuilder::script()->addChild(Js::optimize($script));

        return str_replace('</body>', $htmlScript->render() . "\n" . '</body>', $html);
    }

    public function onAjaxGetformtoken($event): void
    {
        $event->addArgument('result', [Session::getFormToken()]);
    }

    private function addUpdateHitScript(string $body): string
    {
        $input = $this->app->getInput();
        $option = $input->getCmd('option');
        $view = $input->getCmd('view');
        $id = $input->getCmd('id');

        if (
            $id
            && in_array($option, ['com_content', 'com_contact', 'com_tags', 'com_newsfeed'])
            && in_array($view, ['category', 'article', 'tags', 'contacts'])
            && $this->recordHitsOptionSet($option)
        ) {
            $script = $this->getUpdateHitsScript($option, $view, $id);
            $body = str_replace('</body>', "{$script}\n</body>", $body);
        }

        return $body;
    }

    private function getUpdateHitsScript(string $option, string $view, string $id): string
    {
        $baseUrl = SystemUri::siteBaseFull($this->container->get(PathsInterface::class));
        /** @see self::onAjaxUpdatehits() */
        $ajaxPath = 'index.php?option=com_ajax&format=json&plugin=updatehits';
        $url = "{$baseUrl}{$ajaxPath}&hitoption={$option}&hitview={$view}&hitid={$id}";

        return "<script>fetch('{$url}');</script>";
    }

    public function onAjaxUpdatehits(): void
    {
        $input = $this->app->getInput();
        $input->set('hitcount', 1);
        $view = rtrim($input->getCmd('hitview'));
        try {
            $model = $this->app->bootComponent($input->get('hitoption'))
                ->getMVCFactory()
                ->createModel(ucfirst($view), 'Site');
            $model->hit((int)$input->getCmd('hitid'));
        } catch (Exception $e) {
        }
    }

    private function recordHitsOptionSet(string $option): bool
    {
        $recordHits = ComponentHelper::getParams($option)->get('record_hits');

        return $recordHits === null || $recordHits;
    }
}
